import { Link, useLocation } from 'react-router-dom';
import { useState } from 'react';
import { useAuth } from '../contexts/AuthContext';

export default function Sidebar({ collapsed }) {
  const location = useLocation();
  const { user } = useAuth();
  
  // State tracking for collapsible sections
  const [sections, setSections] = useState({
    manage: true,
    configure: false
  });

  const toggleSection = (section) => {
    setSections({
      ...sections,
      [section]: !sections[section]
    });
  };

  // Determine if a link is active
  const isActive = (path) => {
    return location.pathname === path || location.pathname.startsWith(`${path}/`);
  };

  // Check if user has admin role - sử dụng biến này để hiển thị menu theo quyền
  const isAdmin = user?.roles?.includes('admin');

  return (
    <aside 
      className={`${
        collapsed ? 'w-16' : 'w-64'
      } fixed left-0 top-0 h-full bg-gray-900 text-white transition-all duration-300 overflow-y-auto z-20`}
    >
      {/* Logo */}
      <div className={`flex items-center h-16 px-4 ${collapsed ? 'justify-center' : ''}`}>
        {collapsed ? (
          <div className="text-xl font-bold text-blue-400">A</div>
        ) : (
          <Link to="/" className="text-xl font-bold text-blue-400">AuthServer</Link>
        )}
      </div>

      {/* Realm selector */}
      <div className={`${collapsed ? 'px-2' : 'px-4'} py-2`}>
        <div className={`${collapsed ? 'justify-center' : 'justify-between'} flex items-center p-2 bg-gray-800 rounded`}>
          {!collapsed && <span>master</span>}
          <svg className="w-4 h-4" viewBox="0 0 20 20" fill="currentColor">
            <path fillRule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clipRule="evenodd" />
          </svg>
        </div>
      </div>

      {/* Navigation */}
      <nav className="mt-4">
        {/* Manage Section - Chỉ hiển thị đầy đủ cho admin */}
        <div>
          <button 
            onClick={() => toggleSection('manage')}
            className={`${collapsed ? 'justify-center' : 'justify-between'} w-full flex items-center px-4 py-2 text-sm text-gray-300`}
          >
            {!collapsed && <span className="font-semibold">Manage</span>}
            <svg 
              className={`w-4 h-4 transition-transform ${sections.manage ? 'rotate-0' : '-rotate-90'} ${collapsed ? '' : 'ml-auto'}`}
              fill="none" 
              stroke="currentColor" 
              viewBox="0 0 24 24"
            >
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
            </svg>
          </button>
          
          {sections.manage && (
            <div className={`${collapsed ? 'items-center' : ''} flex flex-col py-2`}>
              {isAdmin && (
                <NavItem 
                  to="/admin/realms"
                  icon={(
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 6l3 1m0 0l-3 9a5.002 5.002 0 006.001 0M6 7l3 9M6 7l6-2m6 2l3-1m-3 1l-3 9a5.002 5.002 0 006.001 0M18 7l3 9m-3-9l-6-2m0-2v2m0 16V5m0 16H9m3 0h3" />
                    </svg>
                  )}
                  label="Realms"
                  collapsed={collapsed}
                  active={isActive('/admin/realms')}
                />
              )}
              
              <NavItem 
                to="/admin/clients"
                icon={(
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                  </svg>
                )}
                label="Clients"
                collapsed={collapsed}
                active={isActive('/admin/clients')}
              />
              
              <NavItem 
                to="/admin/users"
                icon={(
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z" />
                  </svg>
                )}
                label="Users"
                collapsed={collapsed}
                active={isActive('/admin/users')}
              />

              <NavItem 
                to="/admin/groups"
                icon={(
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" />
                  </svg>
                )}
                label="Groups"
                collapsed={collapsed}
                active={isActive('/admin/groups')}
              />

              <NavItem 
                to="/admin/roles"
                icon={(
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z" />
                  </svg>
                )}
                label="Roles"
                collapsed={collapsed}
                active={isActive('/admin/roles')}
              />

              <NavItem 
                to="/admin/sessions"
                icon={(
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
                  </svg>
                )}
                label="Sessions"
                collapsed={collapsed}
                active={isActive('/admin/sessions')}
              />

              <NavItem 
                to="/admin/events"
                icon={(
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2" />
                  </svg>
                )}
                label="Events"
                collapsed={collapsed}
                active={isActive('/admin/events')}
              />
            </div>
          )}
        </div>

        {/* Configure Section - Chỉ hiển thị cho admin */}
        {isAdmin && (
          <div>
            <button 
              onClick={() => toggleSection('configure')}
              className={`${collapsed ? 'justify-center' : 'justify-between'} w-full flex items-center px-4 py-2 text-sm text-gray-300`}
            >
              {!collapsed && <span className="font-semibold">Configure</span>}
              <svg 
                className={`w-4 h-4 transition-transform ${sections.configure ? 'rotate-0' : '-rotate-90'} ${collapsed ? '' : 'ml-auto'}`}
                fill="none" 
                stroke="currentColor" 
                viewBox="0 0 24 24"
              >
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
              </svg>
            </button>
            
            {sections.configure && (
              <div className={`${collapsed ? 'items-center' : ''} flex flex-col py-2`}>
                <NavItem 
                  to="/admin/realm-settings"
                  icon={(
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                    </svg>
                  )}
                  label="Realm settings"
                  collapsed={collapsed}
                  active={isActive('/admin/realm-settings')}
                />
                
                <NavItem 
                  to="/admin/authentication"
                  icon={(
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
                    </svg>
                  )}
                  label="Authentication"
                  collapsed={collapsed}
                  active={isActive('/admin/authentication')}
                />
                
                <NavItem 
                  to="/admin/identity-providers"
                  icon={(
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1" />
                    </svg>
                  )}
                  label="Identity providers"
                  collapsed={collapsed}
                  active={isActive('/admin/identity-providers')}
                />
                
                <NavItem 
                  to="/admin/user-federation"
                  icon={(
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z" />
                    </svg>
                  )}
                  label="User federation"
                  collapsed={collapsed}
                  active={isActive('/admin/user-federation')}
                />
              </div>
            )}
          </div>
        )}
      </nav>
    </aside>
  );
}

// Helper component for sidebar navigation items
function NavItem({ to, icon, label, collapsed, active }) {
  return (
    <Link
      to={to}
      className={`
        flex items-center ${collapsed ? 'justify-center' : ''} px-4 py-2 text-sm 
        ${active ? 'bg-gray-800 text-white' : 'text-gray-400 hover:text-white hover:bg-gray-800'}
        ${collapsed ? 'mx-2 rounded-md' : ''}
      `}
    >
      <div className="flex items-center">
        {icon}
        {!collapsed && <span className="ml-3">{label}</span>}
      </div>
    </Link>
  );
}