import { useAuth } from '../contexts/AuthContext';
import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';

export default function Dashboard() {
  const { user } = useAuth();
  const [stats, setStats] = useState({
    users: 0,
    sessions: 0,
    clients: 0,
    events: 0
  });
  const [recentActivity, setRecentActivity] = useState([]);
  
  // Hàm kiểm tra và hiển thị tên người dùng an toàn
  const getUserDisplayName = () => {
    if (!user) return 'Guest';
    return user.username || user.sub || 'User';
  };
  
  useEffect(() => {
    // Simulate data fetching
    setStats({
      users: 24,
      sessions: 12,
      clients: 6,
      events: 147
    });
    
    setRecentActivity([
      { id: 1, type: 'LOGIN', user: 'admin', time: '2025-04-24T10:23:45', ip: '192.168.1.100' },
      { id: 2, type: 'USER_UPDATE', user: 'john.doe', time: '2025-04-24T09:18:32', ip: '192.168.1.101' },
      { id: 3, type: 'CLIENT_CREATE', user: 'admin', time: '2025-04-23T16:45:12', ip: '192.168.1.100' },
      { id: 4, type: 'LOGIN', user: 'jane.smith', time: '2025-04-23T14:12:05', ip: '192.168.1.102' },
      { id: 5, type: 'LOGOUT', user: 'john.doe', time: '2025-04-23T11:32:18', ip: '192.168.1.101' }
    ]);
  }, []);

  return (
    <div className="space-y-8">
      <div>
        <h1 className="text-2xl font-semibold mb-2">
          Welcome{user ? `, ${getUserDisplayName()}` : ' to AuthServer'}
        </h1>
        <p className="text-gray-600">
          Secure authentication and identity management for your applications
        </p>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <StatCard 
          title="Users" 
          count={stats.users} 
          icon={
            <svg className="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z" />
            </svg>
          }
          link="/admin/users"
        />
        <StatCard 
          title="Active Sessions" 
          count={stats.sessions} 
          icon={
            <svg className="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
            </svg>
          }
          link="/admin/sessions"
        />
        <StatCard 
          title="Clients" 
          count={stats.clients} 
          icon={
            <svg className="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
            </svg>
          }
          link="/admin/clients"
        />
        <StatCard 
          title="Recent Events" 
          count={stats.events} 
          icon={
            <svg className="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2" />
            </svg>
          }
          link="/admin/events"
        />
      </div>

      {/* Recent Activity */}
      <div className="bg-white rounded-lg shadow overflow-hidden">
        <div className="px-6 py-4 border-b border-gray-200">
          <h2 className="text-lg font-medium text-gray-900">Recent Activity</h2>
        </div>
        <div className="divide-y divide-gray-200">
          {recentActivity.map((activity) => (
            <div key={activity.id} className="px-6 py-4">
              <div className="flex items-center justify-between">
                <div>
                  <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800 mr-2">
                    {activity.type}
                  </span>
                  <span className="text-sm text-gray-900">by <b>{activity.user}</b></span>
                </div>
                <div className="text-sm text-gray-500">
                  {new Date(activity.time).toLocaleString()} from {activity.ip}
                </div>
              </div>
            </div>
          ))}
        </div>
        <div className="px-6 py-4 bg-gray-50">
          <Link to="/admin/events" className="text-blue-600 hover:text-blue-800 text-sm font-medium">
            View all events →
          </Link>
        </div>
      </div>
      
      {/* Quick actions */}
      <div className="bg-white rounded-lg shadow overflow-hidden">
        <div className="px-6 py-4 border-b border-gray-200">
          <h2 className="text-lg font-medium text-gray-900">Quick Actions</h2>
        </div>
        <div className="px-6 py-4">
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            <QuickActionCard 
              title="Setup TOTP" 
              description="Enable two-factor authentication"
              icon={
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
                </svg>
              }
              link="/totp"
            />
            <QuickActionCard 
              title="Manage Realms" 
              description="Configure authentication domains"
              icon={
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M3 6l3 1m0 0l-3 9a5.002 5.002 0 006.001 0M6 7l3 9M6 7l6-2m6 2l3-1m-3 1l-3 9a5.002 5.002 0 006.001 0M18 7l3 9m-3-9l-6-2m0-2v2m0 16V5m0 16H9m3 0h3" />
                </svg>
              }
              link="/admin/realms"
            />
            <QuickActionCard 
              title="Create User" 
              description="Add a new user account"
              icon={
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M18 9v3m0 0v3m0-3h3m-3 0h-3m-2-5a4 4 0 11-8 0 4 4 0 018 0zM3 20a6 6 0 0112 0v1H3v-1z" />
                </svg>
              }
              link="/admin/users"
            />
            <QuickActionCard 
              title="Create Client" 
              description="Register a new application"
              icon={
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 9v3m0 0v3m0-3h3m-3 0H9m12 0a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
              }
              link="/admin/clients"
            />
          </div>
        </div>
      </div>
    </div>
  );
}

function StatCard({ title, count, icon, link }) {
  return (
    <Link to={link} className="block bg-white rounded-lg shadow hover:shadow-md transition-shadow">
      <div className="p-6">
        <div className="flex items-center justify-between">
          <div className="text-gray-500">{title}</div>
          <div className="text-blue-600">{icon}</div>
        </div>
        <div className="mt-2 text-3xl font-bold">{count}</div>
      </div>
    </Link>
  );
}

function QuickActionCard({ title, description, icon, link }) {
  return (
    <Link to={link} className="block bg-gray-50 hover:bg-gray-100 p-4 rounded-lg transition-colors">
      <div className="flex flex-col items-center text-center">
        <div className="rounded-full bg-blue-100 p-3 text-blue-600 mb-3">{icon}</div>
        <h3 className="text-sm font-medium">{title}</h3>
        <p className="text-xs text-gray-500 mt-1">{description}</p>
      </div>
    </Link>
  );
}
