// ──────────────────────────────────────────
// src/pages/OTPChallenge.jsx
// ──────────────────────────────────────────
import { useLocation, useNavigate } from 'react-router-dom';
import { useState } from 'react';
import { useAuth } from '../contexts/AuthContext';

export default function OTPChallenge() {
  const location = useLocation();
  const nav = useNavigate();
  const { loginRaw, finishLogin } = useAuth();

  /* form gốc gửi từ trang login */
  const baseForm = location.state?.baseForm;

  /* HOOKS phải ở trên cùng, trước mọi 'return' */
  const [code, setCode] = useState('');
  const [err, setErr] = useState('');
  const [isLoading, setIsLoading] = useState(false);

  /* Không có pending login → thông báo */
  if (!baseForm) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-100">
        <div className="bg-white p-8 rounded-lg shadow-md">
          <div className="flex items-center mb-4 text-red-600">
            <svg className="h-6 w-6 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
            <h2 className="text-lg font-semibold">No Pending Login</h2>
          </div>
          <p className="text-gray-600 mb-4">
            There is no active authentication session requiring two-factor verification.
          </p>
          <button
            onClick={() => nav('/login')}
            className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
          >
            Return to login
          </button>
        </div>
      </div>
    );
  }

  /* Verify OTP */
  const verify = async () => {
    if (!code.trim()) {
      setErr('Please enter the verification code');
      return;
    }
    
    setIsLoading(true);
    setErr('');
    
    try {
      const res = await loginRaw({ ...baseForm, otp: code });
      finishLogin(res); // lưu token, set user
      nav('/'); // ➜ Dashboard
    } catch (ex) {
      if (ex.response?.status === 401) setErr('Incorrect verification code');
      else setErr('Verification failed. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gray-100">
      <div className="w-full max-w-md">
        <div className="text-center mb-6">
          <h1 className="text-3xl font-bold text-blue-600">Two-Factor Authentication</h1>
        </div>
        
        <div className="bg-white p-8 rounded-lg shadow-md space-y-6">
          <div className="text-center">
            <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg className="h-8 w-8 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
              </svg>
            </div>
            <h2 className="text-xl font-semibold text-gray-800">Enter Verification Code</h2>
            <p className="text-gray-600 mt-2">
              Please enter the 6-digit code from your authenticator app
            </p>
          </div>

          <div className="space-y-4">
            <div className="space-y-2">
              <input
                type="text"
                className="w-full px-4 py-3 border border-gray-300 rounded-md text-center tracking-widest text-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="000000"
                maxLength="6"
                value={code}
                onChange={(e) => setCode(e.target.value.replace(/\D+/g, '').slice(0, 6))}
                autoFocus
              />
            </div>

            {err && (
              <div className="bg-red-50 border-l-4 border-red-500 p-4 rounded">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-red-500" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-red-700">{err}</p>
                  </div>
                </div>
              </div>
            )}

            <button
              onClick={verify}
              className={`w-full flex justify-center py-3 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white ${
                isLoading ? 'bg-blue-400' : 'bg-blue-600 hover:bg-blue-700'
              } focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500`}
              disabled={isLoading}
            >
              {isLoading ? (
                <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                  <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                  <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                </svg>
              ) : null}
              {isLoading ? 'Verifying...' : 'Verify'}
            </button>

            <div className="text-center">
              <button
                onClick={() => nav('/login')} 
                className="text-sm text-blue-600 hover:text-blue-800"
              >
                Return to login
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
