import { useEffect, useState } from 'react';
import api from '../api';
import { useAuth } from '../contexts/AuthContext';

export default function TOTPSetup({ onSetupComplete, inModal = false }) {
  const { user } = useAuth();
  const [qr, setQr]   = useState('');
  const [otp, setOtp] = useState('');
  const [msg, setMsg] = useState('');

  useEffect(() => {
    api.get('/auth/totp/setup').then(res => setQr(res.data.qr));
  }, []);

  const verify = async () => {
    try {
      await api.post('/auth/totp/verify', { code: otp });
      setMsg('✅ Two‑factor enabled!');
      
      // If onSetupComplete callback is provided and TOTP was successfully set up
      if (onSetupComplete && typeof onSetupComplete === 'function') {
        // Give the user a moment to see success message
        setTimeout(() => {
          onSetupComplete();
        }, 1500);
      }
    } catch { setMsg('❌ Wrong code'); }
  };

  // If in modal, we use a more compact layout
  if (inModal) {
    return (
      <div className="flex flex-col items-center space-y-4">
        <p className="text-sm text-gray-600 mb-2">
          Scan the QR code with your authenticator app (Google Authenticator, Microsoft Authenticator, etc.)
        </p>
        {qr && <img src={qr} alt="qr" className="w-48 h-48 border" />}
        <input 
          className="w-full px-3 py-2 border border-gray-300 rounded-md"
          placeholder="Enter 6‑digit code" 
          value={otp}
          onChange={e=>setOtp(e.target.value)}
        />
        <button 
          className="w-full bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-md"
          onClick={verify}
        >
          Verify
        </button>
        {msg && <p className="text-center">{msg}</p>}
      </div>
    );
  }

  // Default full-page layout
  return (
    <div className="min-h-screen flex flex-col items-center justify-center space-y-4">
      <h1 className="text-xl font-semibold">Enable TOTP for {user?.sub}</h1>
      {qr && <img src={qr} alt="qr" className="w-48 h-48 border" />}
      <input className="input w-60" placeholder="Enter 6‑digit code" value={otp}
             onChange={e=>setOtp(e.target.value)}/>
      <button className="btn w-60" onClick={verify}>Verify</button>
      {msg && <p>{msg}</p>}
    </div>
  );
}
