import { useState, useEffect } from 'react';
import { useAuth } from '../../contexts/AuthContext';
import TOTPSetup from '../TOTPSetup';

export default function Authentication() {
  const { user } = useAuth();
  const [authMethods, setAuthMethods] = useState({
    password: true,
    totp: false,
    webauthn: false,
    oauth: false
  });
  const [loading, setLoading] = useState(false);
  const [showTotpModal, setShowTotpModal] = useState(false);
  const [currentTab, setCurrentTab] = useState('settings');
  
  // Fetch authentication methods from backend (mocked for now)
  useEffect(() => {
    // In a real implementation, this would fetch from the API
    setLoading(true);
    // Mock API call
    setTimeout(() => {
      setAuthMethods({
        password: true,
        totp: localStorage.getItem('totp_enabled') === 'true',
        webauthn: false,
        oauth: false
      });
      setLoading(false);
    }, 500);
  }, [user]);

  const handleToggleTotp = () => {
    if (authMethods.totp) {
      // If TOTP is already enabled, disable it (would need API call in production)
      setAuthMethods({...authMethods, totp: false});
      localStorage.setItem('totp_enabled', 'false');
    } else {
      // If TOTP is disabled, show setup modal
      setShowTotpModal(true);
    }
  };

  const handleTotpSetupComplete = () => {
    setAuthMethods({...authMethods, totp: true});
    localStorage.setItem('totp_enabled', 'true');
    setShowTotpModal(false);
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <h1 className="text-2xl font-semibold mb-2">Authentication</h1>
        <p className="text-gray-600">
          Configure authentication mechanisms and policies for your realms.
          <a href="#" className="text-blue-500 hover:underline ml-2">
            Learn more <span className="inline-block transform translate-y-px">↗</span>
          </a>
        </p>
      </div>

      {/* Tabs */}
      <div className="border-b border-gray-200">
        <nav className="-mb-px flex">
          <button
            onClick={() => setCurrentTab('settings')}
            className={`mr-8 py-2 px-1 ${
              currentTab === 'settings' 
                ? 'border-b-2 border-blue-500 text-blue-600' 
                : 'text-gray-600 hover:text-gray-800 hover:border-b-2 hover:border-gray-300'
            }`}
          >
            Settings
          </button>
          <button
            onClick={() => setCurrentTab('policies')}
            className={`py-2 px-1 ${
              currentTab === 'policies' 
                ? 'border-b-2 border-blue-500 text-blue-600' 
                : 'text-gray-600 hover:text-gray-800 hover:border-b-2 hover:border-gray-300'
            }`}
          >
            Policies
          </button>
        </nav>
      </div>

      {/* Settings Tab Content */}
      {currentTab === 'settings' && (
        <div className="space-y-6">
          {loading ? (
            <div className="p-6 text-center">Loading authentication settings...</div>
          ) : (
            <>
              {/* Authentication Methods */}
              <div className="bg-white shadow rounded-lg overflow-hidden">
                <div className="px-6 py-4 border-b border-gray-200">
                  <h2 className="text-lg font-medium text-gray-900">Authentication Methods</h2>
                </div>
                <div className="divide-y divide-gray-200">
                  {/* Password */}
                  <div className="px-6 py-4 flex items-center justify-between">
                    <div>
                      <h3 className="text-sm font-medium">Password Authentication</h3>
                      <p className="text-xs text-gray-500 mt-1">
                        Basic username and password authentication
                      </p>
                    </div>
                    <div className="flex items-center">
                      <span className={`mr-3 text-sm ${authMethods.password ? 'text-green-600' : 'text-gray-500'}`}>
                        {authMethods.password ? 'Enabled' : 'Disabled'}
                      </span>
                      <button 
                        className="relative inline-flex h-6 w-11 items-center rounded-full bg-gray-200 disabled:opacity-50"
                        disabled
                      >
                        <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition ${authMethods.password ? 'translate-x-6' : 'translate-x-1'}`} />
                      </button>
                    </div>
                  </div>
                  
                  {/* TOTP */}
                  <div className="px-6 py-4 flex items-center justify-between">
                    <div>
                      <h3 className="text-sm font-medium">Time-based One-Time Password (TOTP)</h3>
                      <p className="text-xs text-gray-500 mt-1">
                        Two-factor authentication using authenticator apps
                      </p>
                    </div>
                    <div className="flex items-center">
                      <span className={`mr-3 text-sm ${authMethods.totp ? 'text-green-600' : 'text-gray-500'}`}>
                        {authMethods.totp ? 'Enabled' : 'Disabled'}
                      </span>
                      <button 
                        className="relative inline-flex h-6 w-11 items-center rounded-full bg-gray-200"
                        onClick={handleToggleTotp}
                      >
                        <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition ${authMethods.totp ? 'translate-x-6 bg-green-600' : 'translate-x-1'}`} />
                      </button>
                    </div>
                  </div>
                  
                  {/* WebAuthn */}
                  <div className="px-6 py-4 flex items-center justify-between">
                    <div>
                      <h3 className="text-sm font-medium">WebAuthn / FIDO2</h3>
                      <p className="text-xs text-gray-500 mt-1">
                        Passwordless authentication using security keys
                      </p>
                    </div>
                    <div className="flex items-center">
                      <span className={`mr-3 text-sm ${authMethods.webauthn ? 'text-green-600' : 'text-gray-500'}`}>
                        {authMethods.webauthn ? 'Enabled' : 'Disabled'}
                      </span>
                      <button 
                        className="relative inline-flex h-6 w-11 items-center rounded-full bg-gray-200"
                      >
                        <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition ${authMethods.webauthn ? 'translate-x-6' : 'translate-x-1'}`} />
                      </button>
                    </div>
                  </div>
                  
                  {/* OAuth/Social */}
                  <div className="px-6 py-4 flex items-center justify-between">
                    <div>
                      <h3 className="text-sm font-medium">OAuth / Social Login</h3>
                      <p className="text-xs text-gray-500 mt-1">
                        Allow login via external OAuth providers
                      </p>
                    </div>
                    <div className="flex items-center">
                      <span className={`mr-3 text-sm ${authMethods.oauth ? 'text-green-600' : 'text-gray-500'}`}>
                        {authMethods.oauth ? 'Enabled' : 'Disabled'}
                      </span>
                      <button 
                        className="relative inline-flex h-6 w-11 items-center rounded-full bg-gray-200"
                      >
                        <span className={`inline-block h-4 w-4 transform rounded-full bg-white transition ${authMethods.oauth ? 'translate-x-6' : 'translate-x-1'}`} />
                      </button>
                    </div>
                  </div>
                </div>
              </div>

              {/* TOTP Settings */}
              <div className="bg-white shadow rounded-lg overflow-hidden">
                <div className="px-6 py-4 border-b border-gray-200">
                  <h2 className="text-lg font-medium text-gray-900">TOTP Settings</h2>
                </div>
                <div className="px-6 py-4 space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                      <label htmlFor="otpType" className="block text-sm font-medium text-gray-700 mb-1">OTP Type</label>
                      <select 
                        id="otpType" 
                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        defaultValue="totp"
                        disabled={!authMethods.totp}
                      >
                        <option value="totp">Time Based (TOTP)</option>
                        <option value="hotp">Counter Based (HOTP)</option>
                      </select>
                    </div>
                    <div>
                      <label htmlFor="otpPolicy" className="block text-sm font-medium text-gray-700 mb-1">OTP Policy</label>
                      <select 
                        id="otpPolicy" 
                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        defaultValue="optional"
                        disabled={!authMethods.totp}
                      >
                        <option value="optional">Optional</option>
                        <option value="required">Required</option>
                        <option value="conditional">Conditional</option>
                      </select>
                    </div>
                  </div>

                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div>
                      <label htmlFor="otpDigits" className="block text-sm font-medium text-gray-700 mb-1">OTP Digits</label>
                      <select 
                        id="otpDigits" 
                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        defaultValue="6"
                        disabled={!authMethods.totp}
                      >
                        <option value="6">6</option>
                        <option value="8">8</option>
                      </select>
                    </div>
                    <div>
                      <label htmlFor="otpAlgo" className="block text-sm font-medium text-gray-700 mb-1">Algorithm</label>
                      <select 
                        id="otpAlgo" 
                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        defaultValue="SHA1"
                        disabled={!authMethods.totp}
                      >
                        <option value="SHA1">SHA1</option>
                        <option value="SHA256">SHA256</option>
                        <option value="SHA512">SHA512</option>
                      </select>
                    </div>
                    <div>
                      <label htmlFor="otpPeriod" className="block text-sm font-medium text-gray-700 mb-1">Period</label>
                      <select 
                        id="otpPeriod" 
                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        defaultValue="30"
                        disabled={!authMethods.totp}
                      >
                        <option value="30">30 seconds</option>
                        <option value="60">60 seconds</option>
                      </select>
                    </div>
                  </div>
                </div>
              </div>
            </>
          )}
        </div>
      )}
      
      {/* Policies Tab Content */}
      {currentTab === 'policies' && (
        <div className="bg-white shadow rounded-lg overflow-hidden">
          <div className="px-6 py-4 border-b border-gray-200">
            <h2 className="text-lg font-medium text-gray-900">Authentication Policies</h2>
          </div>
          <div className="px-6 py-4 space-y-4">
            <div>
              <label htmlFor="passwordPolicy" className="block text-sm font-medium text-gray-700 mb-1">Password Policy</label>
              <select 
                id="passwordPolicy" 
                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                defaultValue="default"
              >
                <option value="default">Default Policy</option>
                <option value="strict">Strict Policy</option>
                <option value="custom">Custom Policy</option>
              </select>
            </div>
            
            <div>
              <label htmlFor="lockoutPolicy" className="block text-sm font-medium text-gray-700 mb-1">Account Lockout Policy</label>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label htmlFor="failedAttempts" className="block text-xs text-gray-500 mb-1">Failed Login Attempts</label>
                  <input 
                    id="failedAttempts"
                    type="number"
                    className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                    defaultValue="3"
                  />
                </div>
                <div>
                  <label htmlFor="lockoutDuration" className="block text-xs text-gray-500 mb-1">Lockout Duration (minutes)</label>
                  <input 
                    id="lockoutDuration"
                    type="number"
                    className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                    defaultValue="15"
                  />
                </div>
              </div>
            </div>
            
            <div className="flex items-center mt-4">
              <input
                id="mfaRequired"
                type="checkbox"
                className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
              />
              <label htmlFor="mfaRequired" className="ml-2 text-sm font-medium text-gray-700">
                Require Multi-Factor Authentication (MFA) for all users
              </label>
            </div>
          </div>
        </div>
      )}

      {/* TOTP Setup Modal */}
      {showTotpModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg shadow-lg w-full max-w-md overflow-hidden">
            <div className="px-6 py-4 border-b border-gray-200 flex justify-between items-center">
              <h3 className="text-lg font-medium text-gray-900">Setup Two-Factor Authentication</h3>
              <button 
                onClick={() => setShowTotpModal(false)}
                className="text-gray-400 hover:text-gray-500"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>
            <div className="p-6">
              <TOTPSetup onSetupComplete={handleTotpSetupComplete} inModal={true} />
            </div>
          </div>
        </div>
      )}
    </div>
  );
}