import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import api from '../../api';

export default function Clients() {
  const [clients, setClients] = useState([]);
  const [loading, setLoading] = useState(true);
  const [currentTab, setCurrentTab] = useState('clientsList');
  const [searchTerm, setSearchTerm] = useState('');

  useEffect(() => {
    // Fetch clients from the API
    const fetchClients = async () => {
      try {
        // This endpoint may need to be created on your backend
        const response = await api.get('/admin/clients');
        setClients(response.data || []);
      } catch (error) {
        console.error('Error fetching clients:', error);
        // Fallback with example data if API doesn't exist yet
        setClients([
          {
            id: '1',
            client_id: 'account',
            description: null,
            url: 'https://localhost:8443/realms/Demo/account/',
            type: 'OpenID Connect'
          },
          {
            id: '2',
            client_id: 'account-console',
            description: null,
            url: 'https://localhost:8443/realms/Demo/account/',
            type: 'OpenID Connect'
          },
          {
            id: '3',
            client_id: 'admin-cli',
            description: null,
            url: null,
            type: 'OpenID Connect'
          },
          {
            id: '4',
            client_id: 'broker',
            description: null,
            url: null,
            type: 'OpenID Connect'
          },
          {
            id: '5',
            client_id: 'realm-management',
            description: null,
            url: null,
            type: 'OpenID Connect'
          },
          {
            id: '6',
            client_id: 'security-admin-console',
            description: null,
            url: 'https://localhost:8443/admin/Demo/console/',
            type: 'OpenID Connect'
          }
        ]);
      } finally {
        setLoading(false);
      }
    };

    fetchClients();
  }, []);

  // Filter clients based on search term
  const filteredClients = clients.filter(client => 
    client.client_id.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handleSearch = (e) => {
    setSearchTerm(e.target.value);
  };

  const handleSearchSubmit = (e) => {
    e.preventDefault();
    // Implement additional search functionality if needed
  };

  const handleCreateClient = () => {
    // Implement client creation functionality
    alert('Create client functionality to be implemented');
  };

  const handleImportClient = () => {
    // Implement client import functionality
    alert('Import client functionality to be implemented');
  };

  return (
    <div className="space-y-6">
      {/* Header with title and description */}
      <div>
        <h1 className="text-2xl font-semibold mb-2">Clients</h1>
        <p className="text-gray-600">
          Clients are applications and services that can request authentication of a user.
          <a href="#" className="text-blue-500 hover:underline ml-2">
            Learn more <span className="inline-block transform translate-y-px">↗</span>
          </a>
        </p>
      </div>
      
      {/* Tabs */}
      <div className="border-b border-gray-200">
        <nav className="-mb-px flex">
          <button
            onClick={() => setCurrentTab('clientsList')}
            className={`mr-8 py-2 px-1 ${
              currentTab === 'clientsList' 
                ? 'border-b-2 border-blue-500 text-blue-600' 
                : 'text-gray-600 hover:text-gray-800 hover:border-b-2 hover:border-gray-300'
            }`}
          >
            Clients list
          </button>
          <button
            onClick={() => setCurrentTab('initialAccessToken')}
            className={`py-2 px-1 ${
              currentTab === 'initialAccessToken' 
                ? 'border-b-2 border-blue-500 text-blue-600' 
                : 'text-gray-600 hover:text-gray-800 hover:border-b-2 hover:border-gray-300'
            }`}
          >
            Initial access token
          </button>
        </nav>
      </div>

      {currentTab === 'clientsList' && (
        <div>
          {/* Search and actions */}
          <div className="flex items-center justify-between mb-4">
            <form onSubmit={handleSearchSubmit} className="flex items-center">
              <div className="relative">
                <input
                  type="text"
                  placeholder="Search for client"
                  value={searchTerm}
                  onChange={handleSearch}
                  className="pl-10 pr-4 py-2 border rounded-md w-64 focus:outline-none focus:ring-2 focus:ring-blue-500"
                />
                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                  <svg className="w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                  </svg>
                </div>
              </div>
              <button 
                type="submit"
                className="ml-2 px-3 py-2 border border-gray-300 rounded-md bg-white text-gray-700 hover:bg-gray-50"
              >
                <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M14 5l7 7m0 0l-7 7m7-7H3" />
                </svg>
              </button>
            </form>

            <div className="flex space-x-2">
              <button 
                onClick={handleCreateClient}
                className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
              >
                Create client
              </button>
              <button 
                onClick={handleImportClient}
                className="px-4 py-2 border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
              >
                Import client
              </button>
            </div>
          </div>

          {/* Clients table */}
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Client ID
                  </th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Type
                  </th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Description
                  </th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Home URL
                  </th>
                  <th scope="col" className="relative px-6 py-3">
                    <span className="sr-only">Actions</span>
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {loading ? (
                  <tr>
                    <td colSpan="5" className="px-6 py-4 text-center">
                      Loading...
                    </td>
                  </tr>
                ) : filteredClients.length === 0 ? (
                  <tr>
                    <td colSpan="5" className="px-6 py-4 text-center">
                      No clients found
                    </td>
                  </tr>
                ) : (
                  filteredClients.map((client) => (
                    <tr key={client.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap text-blue-600 hover:underline">
                        <Link to={`/admin/clients/${client.id}`}>
                          {client.client_id}
                        </Link>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        {client.type || '-'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        {client.description || '-'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        {client.url ? (
                          <a 
                            href={client.url} 
                            className="text-blue-600 hover:underline flex items-center"
                            target="_blank"
                            rel="noopener noreferrer"
                          >
                            {client.url}
                            <svg className="w-4 h-4 ml-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M10 6H6a2 2 0 00-2 2v10a2 2 0 002 2h10a2 2 0 002-2v-4M14 4h6m0 0v6m0-6L10 14" />
                            </svg>
                          </a>
                        ) : '-'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                        <button className="text-gray-400 hover:text-gray-500 focus:outline-none">
                          <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 12h.01M12 12h.01M19 12h.01M6 12a1 1 0 11-2 0 1 1 0 012 0zm7 0a1 1 0 11-2 0 1 1 0 012 0zm7 0a1 1 0 11-2 0 1 1 0 012 0z" />
                          </svg>
                        </button>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
          
          {/* Pagination */}
          <div className="flex items-center justify-between mt-6">
            <div className="text-sm text-gray-700">
              Showing 1 - {filteredClients.length} of {filteredClients.length} clients
            </div>
            <div className="flex space-x-2">
              <button className="px-2 py-1 border rounded disabled:opacity-50" disabled>
                &lt;
              </button>
              <button className="px-2 py-1 border rounded disabled:opacity-50" disabled>
                &gt;
              </button>
            </div>
          </div>
        </div>
      )}

      {currentTab === 'initialAccessToken' && (
        <div className="bg-white rounded-md shadow p-6">
          <p className="mb-4">
            Initial access tokens can be used to register new clients through the account service. They can be used only once within their lifespan.
          </p>
          <button 
            className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
          >
            Create initial access token
          </button>
        </div>
      )}
    </div>
  );
}