import { useParams, Link } from 'react-router-dom';
import { useEffect, useState } from 'react';
import { listKeys, createKey } from '../../api/admin';

export default function RealmKeys() {
  const { realmId } = useParams();
  const [keys, setKeys] = useState([]);
  const [loading, setLoading] = useState(true);
  const [realmName, setRealmName] = useState('');
  
  const fetchKeys = async () => {
    setLoading(true);
    try {
      const response = await listKeys(realmId);
      setKeys(response.data);
      
      // Attempt to fetch realm name (assuming your API supports this)
      try {
        const realmResponse = await fetch(`http://localhost:3000/admin/realms/${realmId}`);
        const realmData = await realmResponse.json();
        setRealmName(realmData.name || 'Unknown Realm');
      } catch {
        setRealmName('Unknown Realm');
      }
    } catch (error) {
      console.error('Error fetching keys:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchKeys();
  }, [realmId]);

  const handleGenerateKey = async () => {
    try {
      await createKey(realmId);
      fetchKeys();
    } catch (error) {
      console.error('Error generating key:', error);
    }
  };

  return (
    <div className="space-y-6">
      {/* Header with breadcrumbs */}
      <div>
        <div className="flex items-center text-sm text-gray-500 mb-4">
          <Link to="/admin/realms" className="hover:text-gray-700">Realms</Link>
          <svg className="mx-2 w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M9 5l7 7-7 7" />
          </svg>
          <span>{realmName}</span>
          <svg className="mx-2 w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M9 5l7 7-7 7" />
          </svg>
          <span className="font-medium text-gray-700">Keys</span>
        </div>
        
        <h1 className="text-2xl font-semibold mb-2">Realm Keys</h1>
        <p className="text-gray-600">
          Manage cryptographic keys for this realm. These keys are used for signing tokens and other security operations.
        </p>
      </div>

      {/* Action buttons */}
      <div>
        <button 
          onClick={handleGenerateKey}
          className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
        >
          Generate new key
        </button>
      </div>

      {/* Keys table */}
      <div className="bg-white shadow rounded-lg overflow-hidden">
        {loading ? (
          <div className="p-6 text-center">Loading keys...</div>
        ) : keys.length === 0 ? (
          <div className="p-6 text-center text-gray-500">No keys found</div>
        ) : (
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Key ID (KID)
                </th>
                <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Algorithm
                </th>
                <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Created
                </th>
                <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Expires
                </th>
                <th scope="col" className="relative px-6 py-3">
                  <span className="sr-only">Actions</span>
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {keys.map(key => (
                <tr key={key.id} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="font-mono text-gray-900">{key.kid.slice(0, 8)}...</div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                      {key.alg}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-gray-500">
                    {new Date(key.created_at).toLocaleString()}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-gray-500">
                    {key.expires_at ? new Date(key.expires_at).toLocaleString() : 'No expiration'}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                    <button className="text-gray-400 hover:text-gray-500 focus:outline-none">
                      <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 12h.01M12 12h.01M19 12h.01M6 12a1 1 0 11-2 0 1 1 0 012 0zm7 0a1 1 0 11-2 0 1 1 0 012 0zm7 0a1 1 0 11-2 0 1 1 0 012 0z" />
                      </svg>
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        )}
      </div>
    </div>
  );
}
