// ──────────────────────────────────────────
// prisma/seed.js  –  version fixed
// ──────────────────────────────────────────
const prisma = require('../src/config/prismaClient');
const { hashPassword } = require('../src/utils/passwordUtils');
const { randomUUID } = require('crypto');

async function main() {
  /* 1 ▸ Realm */
  const realm = await prisma.realm.upsert({
    where: { name: 'master' },
    update: {},
    create: {
      id: randomUUID(),
      name: 'master',
      description: 'Default realm',
    },
  });

  /* 2 ▸ Client */
  const client = await prisma.client.upsert({
    where: {
      realm_id_client_id: {
        realm_id: realm.id,
        client_id: 'admin-cli',
      },
    },
    update: {},
    create: {
      id: randomUUID(),
      realm_id: realm.id,
      client_id: 'admin-cli',
      client_name: 'Admin CLI',
      client_secret_hash: '',
      redirect_uris: ['http://localhost:5173/callback'],
      grant_types: ['authorization_code', 'refresh_token', 'password'],
    },
  });

  /* 3 ▸ Role  (find → create nếu chưa có) */
  let role = await prisma.role.findFirst({
    where: {
      realm_id: realm.id,
      client_id: null,
      name: 'admin',
    },
  });

  if (!role) {
    role = await prisma.role.create({
      data: {
        id: randomUUID(),
        realm_id: realm.id,
        client_id: null,      // realm-role
        name: 'admin',
        description: 'Administrator',
      },
    });
  }

  /* 4 ▸ User */
  const { hash, salt, iter } = hashPassword('admin123');
  const user = await prisma.user.upsert({
    where: {
      realm_id_username: { realm_id: realm.id, username: 'admin' },
    },
    update: {},
    create: {
      id: randomUUID(),
      realm_id: realm.id,
      username: 'admin',
      email: 'admin@example.com',
      password_hash: hash,
      password_salt: salt,
      pbkdf2_iter: iter,
      email_verified: true,
    },
  });

  /* 5 ▸ Gán role cho user */
  await prisma.userRole.upsert({
    where: { user_id_role_id: { user_id: user.id, role_id: role.id } },
    update: {},
    create: { user_id: user.id, role_id: role.id },
  });

  console.log('Seed finished ✔');
}

main()
  .catch((e) => {
    console.error(e);
    process.exit(1);
  })
  .finally(() => prisma.$disconnect());
