const asyncHandler = require('express-async-handler');
const prisma = require('../config/prismaClient');
const { hashPassword, verifyPassword } = require('../utils/passwordUtils');
const { generateSecret, otpauthURL, verifyOTP } = require('../services/totpService');
const { signAccessToken, signRefreshToken, rotateRefreshToken } = require('../services/tokenService');
const qrcode = require('qrcode');

exports.register = asyncHandler(async (req, res) => {
  const { realm, username, email, password } = req.body;
  const realmRow = await prisma.realm.findUnique({ where: { name: realm } });
  if (!realmRow) return res.status(400).json({ message: 'Realm not found' });

  const { hash, salt, iter } = hashPassword(password);

  const user = await prisma.user.create({
    data: {
      realm_id: realmRow.id,
      username,
      email,
      password_hash: hash,
      password_salt: salt,
      pbkdf2_iter: iter,
    },
  });

  res.json({ user_id: user.id, message: 'Registered' });
});

exports.login = asyncHandler(async (req, res) => {
  const { realm, username, password, client_id, otp } = req.body;

  const user = await prisma.user.findFirst({
    where: { username, realm: { name: realm } },
    include: {
      roles: { include: { role: true } },
    },
  });

  if (!user) {
    return res.status(401).json({ message: 'Bad credentials' });
  }

  if (!verifyPassword(password, {
    hash: user.password_hash,
    salt: user.password_salt,
    iter: user.pbkdf2_iter,
  })) {
    return res.status(401).json({ message: 'Bad credentials' });
  }
  

  if (user.totp_enabled && !otp) {
    return res.status(206).json({ message: 'Need TOTP' });
  }

  if (user.totp_enabled && !verifyOTP(otp, user.totp_secret)) {
    return res.status(401).json({ message: 'Bad OTP' });
  }

  const client = await prisma.client.findFirst({
    where: { client_id, realm: { name: realm } },
  });

  if (!client) {
    return res.status(400).json({ message: 'Client not found' });
  }

  const access = await signAccessToken(user, client);
  const refresh = await signRefreshToken(user, client);

  res.json({
    access_token: access.token,
    expires_in: access.expiresIn,
    refresh_token: refresh.token,
  });
});

exports.totpSetup = asyncHandler(async (req, res) => {
  const user = await prisma.user.findUnique({ where: { id: req.user.sub } });
  if (!user) return res.status(404).json({ message: 'User not found' });
  if (user.totp_enabled) return res.status(400).json({ message: 'Already enabled' });

  const secret = generateSecret();

  await prisma.user.update({
    where: { id: user.id },
    data: { totp_secret: secret },
  });

  const url = otpauthURL(secret, user.username, 'AuthServer');
  const qr = await qrcode.toDataURL(url);

  res.json({ otpauth: url, qr });
});

exports.totpVerify = asyncHandler(async (req, res) => {
  const { code } = req.body;
  const user = await prisma.user.findUnique({ where: { id: req.user.sub } });

  if (!user || !user.totp_secret) {
    return res.status(400).json({ message: 'Run setup first' });
  }

  const ok = verifyOTP(code, user.totp_secret);
  if (!ok) return res.status(400).json({ message: 'Wrong code' });

  await prisma.user.update({
    where: { id: user.id },
    data: { totp_enabled: true },
  });

  res.json({ message: 'TOTP enabled' });
});

exports.refresh = asyncHandler(async (req, res) => {
  const { refresh_token } = req.body;
  const { token: newRefresh } = await rotateRefreshToken(refresh_token);
  res.json({ refresh_token: newRefresh });
});
