// ──────────────────────────────────────────
// src/index.js
// ──────────────────────────────────────────
//   • Khởi động REST (port 3000) & OIDC (port 4000)
//   • Kết nối Prisma
//   • Chuyển client DB thành cấu hình runtime cho oidc-provider
//   • Thêm cấu hình CORS để frontend (5173) truy cập được
// ──────────────────────────────────────────

require('dotenv').config();
const express       = require('express');
const cors          = require('cors');
const { Provider }  = require('oidc-provider');
const prisma        = require('./config/prismaClient');
const PrismaAdapter = require('./adapters/prismaOidcAdapter');
const authRoutes    = require('./routes/authRoutes');
const realmRoutes = require('./routes/admin/realmRoutes');
const keyRoutes   = require('./routes/admin/keyRoutes');
const userRoutes  = require('./routes/admin/userRoutes');


const REST_PORT = 3000;
const OIDC_PORT = 4000;

const app = express();

// ──────────────────────────────────────────
// Middleware
// ──────────────────────────────────────────
app.use(cors({
  origin: 'http://localhost:5173', // Cho phép frontend truy cập
  credentials: true
}));
app.use(express.json());

// ──────────────────────────────────────────
// Routes
// ──────────────────────────────────────────
app.use('/auth', authRoutes);
app.use('/admin', realmRoutes);
app.use('/admin', keyRoutes);
app.use('/admin', userRoutes);
// ──────────────────────────────────────────
// Load Clients từ DB cho oidc-provider
// ──────────────────────────────────────────
async function loadOIDCClients() {
  const dbClients = await prisma.client.findMany();
  return dbClients.map((c) => ({
    client_id:      c.client_id,
    client_name:    c.client_name,
    client_secret:  c.client_secret_hash || undefined,
    redirect_uris:  c.redirect_uris,
    grant_types:    c.grant_types,
    response_types: ['code'],
    token_endpoint_auth_method: 'client_secret_post',
  }));
}

// ──────────────────────────────────────────
// Main: khởi động cả REST API và OIDC Provider
// ──────────────────────────────────────────
async function main() {
  await prisma.$connect();

  const oidc = new Provider(`http://localhost:${REST_PORT}`, {
    adapter: PrismaAdapter,
    clients: await loadOIDCClients(),
    formats: { AccessToken: 'jwt' },
    features: {
      devInteractions: { enabled: false },
      revocation:      { enabled: true },
      introspection:   { enabled: true },
    },
  });

  app.listen(REST_PORT, () =>
    console.log(`✅ REST API       ➜  http://localhost:${REST_PORT}`)
  );

  oidc.listen(OIDC_PORT, () =>
    console.log(`✅ OIDC Endpoints ➜  http://localhost:${OIDC_PORT}`)
  );
}

// ──────────────────────────────────────────
// Run server
// ──────────────────────────────────────────
main().catch((err) => {
  console.error('❌ Server error:', err);
  process.exit(1);
});
