// ──────────────────────────────────────────
// src/models/User.js
// ──────────────────────────────────────────
const prisma                       = require('../config/prismaClient');
const { randomUUID }               = require('crypto');
const { hashPassword }             = require('../utils/passwordUtils');

async function createUser({ realm_id, username, email, password }) {
  const { hash, salt, iter } = hashPassword(password);
  return prisma.user.create({
    data: {
      id: randomUUID(),
      realm_id,
      username,
      email,
      password_hash: hash,
      password_salt: salt,
      pbkdf2_iter: iter,
    },
  });
}

function findByUsername(realm_id, username) {
  return prisma.user.findUnique({
    where: { realm_id_username: { realm_id, username } },
  });
}

function findByEmail(realm_id, email) {
  return prisma.user.findUnique({
    where: { realm_id_email: { realm_id, email } },
  });
}

function updateTotpSecret(user_id, secret) {
  return prisma.user.update({
    where: { id: user_id },
    data: { totp_secret: secret },
  });
}

function enableTotp(user_id) {
  return prisma.user.update({
    where: { id: user_id },
    data: { totp_enabled: true },
  });
}

function updatePassword(user_id, password) {
  const { hash, salt, iter } = hashPassword(password);
  return prisma.user.update({
    where: { id: user_id },
    data: { password_hash: hash, password_salt: salt, pbkdf2_iter: iter },
  });
}

module.exports = {
  createUser,
  findByUsername,
  findByEmail,
  updateTotpSecret,
  enableTotp,
  updatePassword,
};
