const jwt = require('jsonwebtoken');
const prisma = require('../config/prismaClient');
const { ensureRealmKey } = require('./realmKeyService');
const { randomUUID } = require('crypto');

function ttl(seconds) {
  return Math.floor(Date.now() / 1000) + seconds;
}

async function signAccessToken(user, client) {
  const realm_id = user.realm_id;
  const key = await ensureRealmKey(realm_id);

  const payload = {
    sub: user.id,
    iss: process.env.JWT_ISSUER,
    aud: client.client_id,
    realm: realm_id,
    roles: user.roles.map(r => r.role.name), // đã eager load
  };

  const expiresIn = client.access_token_life || 900;
  const token = jwt.sign(payload, key.private_key_enc, {
    algorithm: key.alg,
    expiresIn,
    keyid: key.kid,
    jwtid: randomUUID(),
  });

  await prisma.token.create({
    data: {
      id: randomUUID(),
      user_id: user.id,
      client_id: client.id,
      token_type: 'access',
      jwt_id: jwt.decode(token).jti,
      issued_at: new Date(),
      expires_at: new Date(Date.now() + expiresIn * 1000),
      scope: '', // tuỳ implement
    },
  });

  return { token, expiresIn };
}

async function signRefreshToken(user, client) {
  const realm_id = user.realm_id;
  const key = await ensureRealmKey(realm_id);

  const expiresIn = client.refresh_token_life || 14 * 24 * 3600;
  const payload = { sub: user.id, aud: client.client_id, type: 'refresh' };

  const token = jwt.sign(payload, key.private_key_enc, {
    algorithm: key.alg,
    expiresIn,
    keyid: key.kid,
    jwtid: randomUUID(),
  });

  await prisma.token.create({
    data: {
      id: randomUUID(),
      user_id: user.id,
      client_id: client.id,
      token_type: 'refresh',
      jwt_id: jwt.decode(token).jti,
      issued_at: new Date(),
      expires_at: new Date(Date.now() + expiresIn * 1000),
      scope:'',
    },
  });

  return { token, expiresIn };
}

async function rotateRefreshToken(oldTokenStr) {
  const decoded = jwt.decode(oldTokenStr);
  const dbToken = await prisma.token.findUnique({ where: { jwt_id: decoded.jti } });
  if (!dbToken || dbToken.revoked_at) throw new Error('Invalid refresh token');

  // Revoke cũ
  await prisma.token.update({ where: { id: dbToken.id }, data: { revoked_at: new Date() } });

  // Tạo mới
  const user = await prisma.user.findUnique({ where: { id: dbToken.user_id }, include: { roles: { include: { role: true } } } });
  const client = await prisma.client.findUnique({ where: { id: dbToken.client_id } });
  return signRefreshToken(user, client);
}

module.exports = { signAccessToken, signRefreshToken, rotateRefreshToken };
